/* $Id: server.cpp $ */
/** @file
 * VBox Remote Desktop Protocol Server.
 */

/*
 * Copyright (C) 2006-2025 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * SPDX-License-Identifier: GPL-3.0-only
 */

#include "vrdpserv.h"
#include "tcp.h"

/*
 * The VRDPServer supports many clients.
 *
 * Each client instance has an INPUT and OUTPUT contexts.
 * The INPUT context is responsible for RDP connection
 * establishing and for receiving clients RDP packets.
 * The OUTPUT context is used only for sending the server
 * RDP packets.
 *
 * When a transport connection is established, the server
 * will create a new instance of VRDPClient and map it to the
 * transport specific identifier (socket for TCP transport).
 * The client instance will establish RDP connection on INPUT
 * context. After this is done, the server enables OUTPUT
 * context for the client. Now all output, generated by the
 * server, will go to the client instance as well.
 *
 * When server has something to output, the output thread will call all client
 * instances, which have the OUTPUT context enabled, with the
 * update. The client will use a transport callback to actually
 * send the update, depending on the client state.
 *
 * When there is a RDP packet received by the transport, the server
 * will find the corresponding client instance and let it to process
 * the packet.
 *
 * The server operates on 3 threads:
 *   * VM thread:
 *        All server API are called on the thread, the thread is external.
 *        If the operation requires transport, then it scheduled for
 *        for execution on INPUT or OUTPUT thread. Other operations
 *        are executed directly, guarded by the server lock.
 *        NULL context - no clients instances can be touched at this context.
 *   * INPUT thread:
 *        The server listens on transport connections and manages
 *        clients.
 *        INPUT context - sending can only be done when output context is disabled.
 *   * OUTPUT thread:
 *        The server processes output updates.
 *        OUTPUT context - only sending.
 *
 * INPUT and OUTPUT threads are created at the server startup.
 *
 */

#define VRDP_DEFAULT_PORT 3389

/*
 * Thread functions.
 */
static DECLCALLBACK(int) InputThreadFunc (RTTHREAD self, void *pvUser)
{
    VRDPServerThreadStartCtx *pCtx = (VRDPServerThreadStartCtx *)pvUser;

    return pCtx->pVRDPServer->InputThread (self, pCtx);
}

static DECLCALLBACK(int) OutputThreadFunc (RTTHREAD self, void *pvUser)
{
    VRDPServerThreadStartCtx *pCtx = (VRDPServerThreadStartCtx *)pvUser;

    return pCtx->pVRDPServer->OutputThread (self, pCtx);
}

static void appVideoModeHint(const VRDECALLBACKS_4 *pCallbacks,
                             void *pvCallback,
                             unsigned cWidth,
                             unsigned cHeight,
                             unsigned cBitsPerPixel,
                             unsigned uScreenId)
{
    if (pCallbacks && pCallbacks->VRDECallbackVideoModeHint)
    {
        pCallbacks->VRDECallbackVideoModeHint (pvCallback, cWidth, cHeight, cBitsPerPixel, uScreenId);
    }
}

static int appClientLogon (const VRDECALLBACKS_4 *pCallbacks,
                           void *pvCallback,
                           uint32_t u32ClientId,
                           const char *pszUser,
                           const char *pszPassword,
                           const char *pszDomain)
{
    int rc = VERR_NOT_SUPPORTED;

    if (pCallbacks && pCallbacks->VRDECallbackClientLogon)
    {
        rc = pCallbacks->VRDECallbackClientLogon (pvCallback, u32ClientId, pszUser, pszPassword, pszDomain);
    }

    return rc;
}

static void appClientConnect (const VRDECALLBACKS_4 *pCallbacks,
                              void *pvCallback,
                              uint32_t u32ClientId)
{
    if (pCallbacks && pCallbacks->VRDECallbackClientConnect)
    {
        pCallbacks->VRDECallbackClientConnect (pvCallback, u32ClientId);
    }
}

static void appClientDisconnect (const VRDECALLBACKS_4 *pCallbacks,
                                 void *pvCallback,
                                 uint32_t u32ClientId,
                                 uint32_t fu32Intercepted)
{
    if (pCallbacks && pCallbacks->VRDECallbackClientDisconnect)
    {
        pCallbacks->VRDECallbackClientDisconnect (pvCallback, u32ClientId, fu32Intercepted);
    }
}

static int appIntercept (const VRDECALLBACKS_4 *pCallbacks,
                         void *pvCallback,
                         uint32_t u32ClientId,
                         uint32_t fu32Intercept,
                         void **ppvIntercept)
{
    int rc = VERR_NOT_SUPPORTED;

    if (pCallbacks && pCallbacks->VRDECallbackIntercept)
    {
        rc = pCallbacks->VRDECallbackIntercept (pvCallback, u32ClientId, fu32Intercept, ppvIntercept);
    }

    return rc;
}

static void appInput (const VRDECALLBACKS_4 *pCallbacks,
                      void *pvCallback,
                      int type,
                      const void *pvInput,
                      unsigned cbInput)
{
    if (pCallbacks && pCallbacks->VRDECallbackInput)
    {
        pCallbacks->VRDECallbackInput (pvCallback, type, pvInput, cbInput);
    }
}

static bool appFramebufferQuery (const VRDECALLBACKS_4 *pCallbacks,
                                 void *pvCallback,
                                 unsigned uScreenId,
                                 VRDEFRAMEBUFFERINFO *pInfo)
{
    bool fAvailable = false;

    if (pCallbacks && pCallbacks->VRDECallbackFramebufferQuery)
    {
        fAvailable = pCallbacks->VRDECallbackFramebufferQuery (pvCallback, uScreenId, pInfo);
    }

    return fAvailable;
}

static void appFramebufferLock (const VRDECALLBACKS_4 *pCallbacks,
                                void *pvCallback,
                                unsigned uScreenId)
{
    if (pCallbacks && pCallbacks->VRDECallbackFramebufferLock)
    {
        pCallbacks->VRDECallbackFramebufferLock (pvCallback, uScreenId);
    }
}

static void appFramebufferUnlock (const VRDECALLBACKS_4 *pCallbacks,
                                  void *pvCallback,
                                  unsigned uScreenId)
{
    if (pCallbacks && pCallbacks->VRDECallbackFramebufferUnlock)
    {
        pCallbacks->VRDECallbackFramebufferUnlock (pvCallback, uScreenId);
    }
}

int appProperty (const VRDECALLBACKS_4 *pCallbacks,
                 void *pvCallback,
                 uint32_t index,
                 void *pvBuffer,
                 uint32_t cbBuffer,
                 uint32_t *pcbOut)
{
    int rc = VERR_NOT_SUPPORTED;

    if (pCallbacks && pCallbacks->VRDECallbackProperty)
    {
        if (pcbOut == NULL)
        {
            /* A fixed length value is queried. */
            uint32_t cbOut = 0;
            rc = pCallbacks->VRDECallbackProperty (pvCallback, index, pvBuffer, cbBuffer, &cbOut);
            if (RT_SUCCESS (rc) && cbBuffer != cbOut)
            {
                /* Should not actually happen. */
                rc = VERR_INVALID_PARAMETER;
            }
        }
        else
        {
            /* A variable length value is queried. The 'pvBuffer' points to a pointer to a value.
             * The 'cbBuffer' is the length of the value, if the pointer to value is not NULL.
             */
            void *pvValue = *(void **)pvBuffer;
            if (pvValue != NULL)
            {
                /* The caller provided a buffer for the value. */
                rc = pCallbacks->VRDECallbackProperty (pvCallback, index, pvValue, cbBuffer, pcbOut);
                if (rc == VINF_BUFFER_OVERFLOW)
                {
                    rc = VERR_BUFFER_OVERFLOW;
                }
            }
            else
            {
                /* The caller wants the value buffer to be allocated for him.
                 * First query the value length.
                 */
                uint32_t cbOut = 0;
                rc = pCallbacks->VRDECallbackProperty (pvCallback, index, NULL, 0, &cbOut);

                if (RT_SUCCESS(rc))
                {
                    /* 'cbOut' is the length of the value. */
                    if (cbOut > 0)
                    {
                        /* Allocate the memory buffer and query the value. */
                        pvValue = VRDPMemAllocZ (cbOut);
                        if (pvValue == NULL)
                        {
                            rc = VERR_NO_MEMORY;
                        }
                        else
                        {
                            rc = pCallbacks->VRDECallbackProperty (pvCallback, index, pvValue, cbOut, &cbOut);
                            if (RT_SUCCESS (rc))
                            {
                                *(void **)pvBuffer = pvValue;
                            }
                        }
                    }

                    if (RT_SUCCESS (rc))
                    {
                        *pcbOut = cbOut;
                    }
                }
            }
        }
    }

    if (RT_FAILURE (rc))
    {
        VRDPLOGREL(("Failed to query a property: %d, rc = %Rrc\n", index, rc));
    }

    return rc;
}

int appFeature (const VRDECALLBACKS_4 *pCallbacks,
                void *pvCallback,
                const char *pszName,
                char **ppszValue,
                uint32_t *pcbOut)
{
    if (!ppszValue)
    {
        return VERR_INVALID_PARAMETER;
    }

    uint8_t au8Buffer[4096];

    VRDEFEATURE *pFeature = (VRDEFEATURE *)&au8Buffer[0];

    int rc = RTStrCopy(pFeature->achInfo, sizeof (au8Buffer) - RT_UOFFSETOF(VRDEFEATURE, achInfo), pszName);

    if (RT_SUCCESS(rc))
    {
        pFeature->u32ClientId = 0;

        uint32_t cbOut = 0;
        rc = appProperty(pCallbacks, pvCallback,
                         VRDE_QP_FEATURE,
                         &pFeature, sizeof (au8Buffer), &cbOut);

        if (RT_SUCCESS(rc))
        {
            if (cbOut > 0)
            {
                char *pszValue = (char *)VRDPMemAlloc(cbOut);

                if (pszValue)
                {
                    memcpy(pszValue, &pFeature->achInfo, cbOut);

                    *ppszValue = pszValue;
                    if (pcbOut)
                    {
                        *pcbOut = cbOut;
                    }
                }
                else
                {
                    rc = VERR_NO_MEMORY;
                }
            }
            else
            {
                rc = VERR_NOT_SUPPORTED;
            }
        }
    }

    if (RT_FAILURE (rc))
    {
        VRDPLOGREL(("Failed to query [%s]: rc = %Rrc\n", pszName, rc));
    }

    return rc;
}

/*
 * VRDP server implementation.
 */

/*
 * If stats are enabled, call the collector.
 */
static int vhStatInit(VHSTAT *pStat)
{
    int rc = VHStatCreate(&pStat->pStat, VH_STAT_FULL);

    if (RT_SUCCESS(rc))
    {
        VRDPLOGREL(("Statistics created: [%s], enabled: %d.\n",
                     pStat->pStat->Description(),
                     LogRelIs6Enabled()));
    }
    else
    {
        VRDPLOGREL(("Statistics not created %Rrc.\n",
                     rc));
    }

    return rc;
}

static void vhStatDestroy(VHSTAT *pStat)
{
    VHStatDelete(pStat->pStat);
}


static void copyCallbacks(VRDECALLBACKS_4 *pApplicationCallbacks, int callbackVersion, const VRDEINTERFACEHDR *pCallbackHdr)
{
    memset (pApplicationCallbacks, 0, sizeof(VRDECALLBACKS_4));

    switch(callbackVersion)
    {
        case 1:
        {
           const VRDECALLBACKS_1 *p = (VRDECALLBACKS_1 *)pCallbackHdr;

           pApplicationCallbacks->header = p->header;
           pApplicationCallbacks->VRDECallbackProperty = p->VRDECallbackProperty;
           pApplicationCallbacks->VRDECallbackClientLogon = p->VRDECallbackClientLogon;
           pApplicationCallbacks->VRDECallbackClientConnect = p->VRDECallbackClientConnect;
           pApplicationCallbacks->VRDECallbackClientDisconnect = p->VRDECallbackClientDisconnect;
           pApplicationCallbacks->VRDECallbackIntercept = p->VRDECallbackIntercept;
           pApplicationCallbacks->VRDECallbackUSB = p->VRDECallbackUSB;
           pApplicationCallbacks->VRDECallbackClipboard = p->VRDECallbackClipboard;
           pApplicationCallbacks->VRDECallbackFramebufferQuery = p->VRDECallbackFramebufferQuery;
           pApplicationCallbacks->VRDECallbackFramebufferLock = p->VRDECallbackFramebufferLock;
           pApplicationCallbacks->VRDECallbackFramebufferUnlock = p->VRDECallbackFramebufferUnlock;
           pApplicationCallbacks->VRDECallbackInput = p->VRDECallbackInput;
           pApplicationCallbacks->VRDECallbackVideoModeHint = p->VRDECallbackVideoModeHint;
        } break;
        case 2:
        {
           const VRDECALLBACKS_2 *p = (VRDECALLBACKS_2 *)pCallbackHdr;

           pApplicationCallbacks->header = p->header;
           pApplicationCallbacks->VRDECallbackProperty = p->VRDECallbackProperty;
           pApplicationCallbacks->VRDECallbackClientLogon = p->VRDECallbackClientLogon;
           pApplicationCallbacks->VRDECallbackClientConnect = p->VRDECallbackClientConnect;
           pApplicationCallbacks->VRDECallbackClientDisconnect = p->VRDECallbackClientDisconnect;
           pApplicationCallbacks->VRDECallbackIntercept = p->VRDECallbackIntercept;
           pApplicationCallbacks->VRDECallbackUSB = p->VRDECallbackUSB;
           pApplicationCallbacks->VRDECallbackClipboard = p->VRDECallbackClipboard;
           pApplicationCallbacks->VRDECallbackFramebufferQuery = p->VRDECallbackFramebufferQuery;
           pApplicationCallbacks->VRDECallbackFramebufferLock = p->VRDECallbackFramebufferLock;
           pApplicationCallbacks->VRDECallbackFramebufferUnlock = p->VRDECallbackFramebufferUnlock;
           pApplicationCallbacks->VRDECallbackInput = p->VRDECallbackInput;
           pApplicationCallbacks->VRDECallbackVideoModeHint = p->VRDECallbackVideoModeHint;
        } break;
        case 3:
        {
           const VRDECALLBACKS_3 *p = (VRDECALLBACKS_3 *)pCallbackHdr;

           pApplicationCallbacks->header = p->header;
           pApplicationCallbacks->VRDECallbackProperty = p->VRDECallbackProperty;
           pApplicationCallbacks->VRDECallbackClientLogon = p->VRDECallbackClientLogon;
           pApplicationCallbacks->VRDECallbackClientConnect = p->VRDECallbackClientConnect;
           pApplicationCallbacks->VRDECallbackClientDisconnect = p->VRDECallbackClientDisconnect;
           pApplicationCallbacks->VRDECallbackIntercept = p->VRDECallbackIntercept;
           pApplicationCallbacks->VRDECallbackUSB = p->VRDECallbackUSB;
           pApplicationCallbacks->VRDECallbackClipboard = p->VRDECallbackClipboard;
           pApplicationCallbacks->VRDECallbackFramebufferQuery = p->VRDECallbackFramebufferQuery;
           pApplicationCallbacks->VRDECallbackFramebufferLock = p->VRDECallbackFramebufferLock;
           pApplicationCallbacks->VRDECallbackFramebufferUnlock = p->VRDECallbackFramebufferUnlock;
           pApplicationCallbacks->VRDECallbackInput = p->VRDECallbackInput;
           pApplicationCallbacks->VRDECallbackVideoModeHint = p->VRDECallbackVideoModeHint;
           pApplicationCallbacks->VRDECallbackAudioIn = p->VRDECallbackAudioIn;
        } break;
        case 4:
        {
           const VRDECALLBACKS_4 *p = (VRDECALLBACKS_4 *)pCallbackHdr;

           pApplicationCallbacks->header = p->header;
           pApplicationCallbacks->VRDECallbackProperty = p->VRDECallbackProperty;
           pApplicationCallbacks->VRDECallbackClientLogon = p->VRDECallbackClientLogon;
           pApplicationCallbacks->VRDECallbackClientConnect = p->VRDECallbackClientConnect;
           pApplicationCallbacks->VRDECallbackClientDisconnect = p->VRDECallbackClientDisconnect;
           pApplicationCallbacks->VRDECallbackIntercept = p->VRDECallbackIntercept;
           pApplicationCallbacks->VRDECallbackUSB = p->VRDECallbackUSB;
           pApplicationCallbacks->VRDECallbackClipboard = p->VRDECallbackClipboard;
           pApplicationCallbacks->VRDECallbackFramebufferQuery = p->VRDECallbackFramebufferQuery;
           pApplicationCallbacks->VRDECallbackFramebufferLock = p->VRDECallbackFramebufferLock;
           pApplicationCallbacks->VRDECallbackFramebufferUnlock = p->VRDECallbackFramebufferUnlock;
           pApplicationCallbacks->VRDECallbackInput = p->VRDECallbackInput;
           pApplicationCallbacks->VRDECallbackVideoModeHint = p->VRDECallbackVideoModeHint;
           pApplicationCallbacks->VRDECallbackAudioIn = p->VRDECallbackAudioIn;
        } break;
    }
}


/* Constructor, only initialize members. */
VRDPServer::VRDPServer (int callbackVersion, const VRDEINTERFACEHDR *pCallbackHdr, void *pvCallback)
    :
    m_pointerCache (),
    m_outputQueue (),
    m_clientArray (),
    m_scard(this),
    m_tsmf(this),
    m_videoin(this),
    m_input(this)
{
    copyCallbacks(&m_ApplicationCallbacks, callbackVersion, pCallbackHdr);
    m_pApplicationCallbacks = &m_ApplicationCallbacks;
    m_pvApplicationCallback = pvCallback;

    m_fConnectionsEnabled = false;

    m_inputThread  = NIL_RTTHREAD;
    m_outputThread = NIL_RTTHREAD;
    m_fShutdownThreads = false;

    m_hEvtInput    = NIL_RTSEMEVENT;
    m_outputsem    = NIL_RTSEMEVENTMULTI;

    m_pTransport   = NULL;
    m_port         = VRDP_DEFAULT_PORT;

    m_mouseButtons = 0;
    m_mousex       = 0;
    m_mousey       = 0;

    m_fKeyboardExtension = VRDPEnvExist("VBOX_VRDP_KEYBOARD_EXT") || VRDPEnvExist("VRDP_KEYBOARD_EXT");

    m_pbc = NULL;
    m_ptc = NULL;
    u32ClientIdSrc = 0;
    m_u64LastInputTS = 0;
    mu32ClipboardDataWriteClientId = 0;

    m_pszAddress = NULL;
    m_pszPortRange = NULL;
    m_u32BindPort = 0;

    m_pszUnixSocketPath = NULL;

    m_pServerLock = NULL;

    memset (&m_AudioData, 0, sizeof (m_AudioData));

    m_pVideoHandler = NULL;

    m_paFBInfos = NULL;
    m_cMonitors = 0;

    memset (&m_KeyboardModifiers, 0, sizeof (m_KeyboardModifiers));

    memset (&m_imageInterface, 0, sizeof(m_imageInterface));
}

VRDPServer::~VRDPServer (void)
{
    /* Notify about the server shutdown. */
    m_u32BindPort = UINT32_MAX;
    appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                      VRDE_SP_NETWORK_BIND_PORT,
                      &m_u32BindPort, sizeof (m_u32BindPort), NULL);

    m_scard.SCardShutdown();
    m_tsmf.TSMFShutdown();
    m_videoin.VideoInShutdown();
    m_input.InputShutdown();

    ShutdownThreads();

    /* Transport must be deleted after both OUTPUT and INPUT thread has been shutdown. */
    delete m_pTransport;
    m_pTransport = NULL;

    /* No more application callbacks. */
    m_pApplicationCallbacks = NULL;
    m_pvApplicationCallback = NULL;

    videoHandlerUninit();

    shadowBufferDestroyBuffers ();

    shadowBufferUninit ();

    BCDelete (m_pbc);

    if (m_outputsem != NIL_RTSEMEVENTMULTI)
    {
        RTSemEventMultiDestroy (m_outputsem);
        m_outputsem = NIL_RTSEMEVENTMULTI;
    }

    if (m_hEvtInput != NIL_RTSEMEVENT)
    {
        RTSemEventDestroy(m_hEvtInput);
        m_hEvtInput = NIL_RTSEMEVENT;
    }

    VRDPLock::Delete(&m_pServerLock);

    VRDPMemFree (m_paFBInfos);
    m_paFBInfos = NULL;
    if (m_pszAddress)
    {
        VRDPMemFree (m_pszAddress);
        m_pszAddress = NULL;
    }
    if (m_pszUnixSocketPath)
    {
        VRDPMemFree (m_pszUnixSocketPath);
        m_pszUnixSocketPath = NULL;
    }
    if (m_pszPortRange)
    {
        VRDPMemFree (m_pszPortRange);
        m_pszPortRange = NULL;
    }

    vhStatDestroy(&m_stat);
}

int VRDPServer::Enter (void)
{
    return VRDPLock::Lock(m_pServerLock);
}

void VRDPServer::Exit (void)
{
    VRDPLock::Unlock(m_pServerLock);
}


int VRDPServer::GetInterface (const char *pszId,
                              VRDEINTERFACEHDR *pInterface,
                              const VRDEINTERFACEHDR *pCallbacks,
                              void *pvContext)
{
    int rc = VERR_NOT_SUPPORTED;

    if (RTStrICmp(pszId, VRDE_IMAGE_INTERFACE_NAME) == 0)
    {
        /* Verify the requested version and obtain interface pointers. */
        rc = vrdpGetInterfaceImage(pInterface, pCallbacks);

        if (RT_SUCCESS(rc))
        {
            /* Save required information. */
            m_imageInterface.callbacks = *(VRDEIMAGECALLBACKS *)pCallbacks;
            m_imageInterface.pvContext = pvContext;
        }
    }
    else if (RTStrICmp(pszId, VRDE_MOUSEPTR_INTERFACE_NAME) == 0)
    {
        /* Verify the requested version and obtain interface pointers. */
        rc = vrdpGetInterfaceMousePtr(pInterface, pCallbacks);
    }
    else if (RTStrICmp(pszId, VRDE_SCARD_INTERFACE_NAME) == 0)
    {
        /* Verify the requested version and obtain interface pointers. */
        rc = SCard()->GetInterfaceSCard(pInterface, pCallbacks, pvContext);
    }
    else if (RTStrICmp(pszId, VRDE_TSMF_INTERFACE_NAME) == 0)
    {
        /* Verify the requested version and obtain interface pointers. */
        rc = TSMF()->GetInterfaceTSMF(pInterface, pCallbacks, pvContext);
    }
    else if (RTStrICmp(pszId, VRDE_VIDEOIN_INTERFACE_NAME) == 0)
    {
        /* Verify the requested version and obtain interface pointers. */
        rc = VideoIn()->GetInterfaceVideoIn(pInterface, pCallbacks, pvContext);
    }
    else if (RTStrICmp(pszId, VRDE_INPUT_INTERFACE_NAME) == 0)
    {
        rc = Input()->GetInterfaceInput(pInterface, pCallbacks, pvContext);
    }

    return rc;
}

/* Actually initializes the server, obtains configuration
 * from the application and starts threads.
 */

int VRDPServer::Start (void)
{
#ifdef VRDP_ENABLE_LOGREL
    RTLogSetGroupLimit(RTLogRelGetDefaultInstance(), UINT32_MAX);
#endif

    int rc = internalStart();

    if (RT_FAILURE (rc))
    {
        /* Server failed to start. *Set* the bind port property.
         * Note: on successful start the property will be set in
         * EnableConnections.
         */
        m_u32BindPort = 0;
        appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                          VRDE_SP_NETWORK_BIND_PORT,
                          &m_u32BindPort, sizeof (m_u32BindPort), NULL);
    }

    return rc;
}

int VRDPServer::internalStart (void)
{
    int rc = VINF_SUCCESS;

    uint32_t cbOut;

    rc = vhStatInit(&m_stat);

    if (RT_FAILURE(rc))
    {
        return rc;
    }

    rc = VRDPLock::Create("Server", &m_pServerLock);

    if (RT_FAILURE(rc))
    {
        return rc;
    }

    rc = RTSemEventCreate(&m_hEvtInput);
    AssertRCReturnStmt(rc, m_hEvtInput = NIL_RTSEMEVENT, rc);

    rc = RTSemEventMultiCreate(&m_outputsem);
    AssertRCReturnStmt(rc, m_outputsem = NIL_RTSEMEVENTMULTI, rc);

    rc = appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                           VRDE_QP_NUMBER_MONITORS,
                           &m_cMonitors, sizeof (m_cMonitors), NULL);

    if (RT_FAILURE(rc))
    {
        return rc;
    }

    rc = BCCreate (&m_pbc, 8 * _1M);

    if (RT_FAILURE(rc))
    {
        VRDPLOGREL(("Failed to initialize the bitmap cache, rc = %Rrc\n", rc));
        return VERR_NO_MEMORY;
    }

    SERVERLOG(("cMonitors = %d\n", m_cMonitors));

    rc = shadowBufferInit (this, m_cMonitors);

    if (RT_FAILURE(rc))
    {
        VRDPLOGREL(("Failed to initialize the shadow buffer, rc = %Rrc\n", rc));
        return rc;
    }

    uint32_t u32VideoEnabled = 0;
    rc = appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                           VRDE_QP_VIDEO_CHANNEL,
                           &u32VideoEnabled, sizeof (u32VideoEnabled), NULL);

    if (RT_FAILURE(rc))
    {
        u32VideoEnabled = 0;
    }

    u32VideoEnabled = u32VideoEnabled || VRDPEnvExist("VBOX_VRDP_VIDEO");

    if (u32VideoEnabled != 0)
    {
        videoHandlerInit();

    }

    m_paFBInfos = (VRDPFBINFO *)VRDPMemAllocZ (sizeof (VRDPFBINFO) * m_cMonitors);

    if (m_paFBInfos == NULL)
    {
        return VERR_NO_MEMORY;
    }

    cbOut = 0;
    rc = appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                           VRDE_QP_UNIX_SOCKET_PATH,
                           &m_pszUnixSocketPath, 0, &cbOut);

    if (RT_FAILURE(rc))
    {
        m_pszUnixSocketPath = NULL;
    }

    if (m_pszUnixSocketPath != NULL)
    {
        SERVERLOG(("UNIX socket = %s\n", m_pszUnixSocketPath));
    }


    rc = appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                           VRDE_QP_NETWORK_PORT,
                           &m_port, sizeof (m_port), NULL);

    if (RT_FAILURE(rc))
    {
        return rc;
    }

    if (m_port == 0)
    {
        m_port = VRDP_DEFAULT_PORT;
    }

    SERVERLOG(("port = %d\n", m_port));

    cbOut = 0;
    rc = appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                           VRDE_QP_NETWORK_ADDRESS,
                           &m_pszAddress, 0, &cbOut);

    if (RT_FAILURE(rc))
    {
        return rc;
    }

    cbOut = 0;
    rc = appFeature (ApplicationCallbacks (), ApplicationCallbackPointer (),
                     "Property/TCP/Ports",
                     &m_pszPortRange, &cbOut);
    if (RT_FAILURE (rc))
    {
        m_pszPortRange = NULL; /* Make sure it is NULL, if it does not exist. */
    }

    rc = m_scard.SCardInitialize();
    if (RT_FAILURE(rc))
    {
        return rc;
    }

    rc = m_tsmf.TSMFInitialize();
    if (RT_FAILURE(rc))
    {
        return rc;
    }

    rc = m_videoin.VideoInInitialize();
    if (RT_FAILURE(rc))
    {
        return rc;
    }

    rc = m_input.InputInitialize();
    if (RT_FAILURE(rc))
    {
        return rc;
    }

    /* Create the transport here so the treads can access the object. */
    rc = TCPTransportCreate(&m_pTransport, this, m_pszAddress, m_port,
                            m_pszPortRange, m_pszUnixSocketPath);
    if (RT_FAILURE(rc))
    {
        return rc;
    }

    /* Start Input thread, which will also create the transport. */
    rc = StartThread (&m_inputThread, InputThreadFunc, "VRDP-IN");

    if (RT_FAILURE(rc))
    {
        return rc;
    }

    rc = StartThread (&m_outputThread, OutputThreadFunc, "VRDP-OUT");

    if (RT_FAILURE(rc))
    {
        ShutdownThreads ();
        return rc;
    }

    return VINF_SUCCESS;
}

int VRDPServer::EnableConnections (bool fEnable)
{
    if (fEnable)
    {
        /* Pickup the current application framebuffer(s). */
        ProcessResize ();

        m_fConnectionsEnabled = true;

        /* Server started successfully and the application enables connections. *Set* the bind port property. */
        appProperty (ApplicationCallbacks (), ApplicationCallbackPointer (),
                          VRDE_SP_NETWORK_BIND_PORT,
                          &m_u32BindPort, sizeof (m_u32BindPort), NULL);

        /* Kick the thread out of it's "sleep" and make sure it starts listening. */
        int rc = RTSemEventSignal (m_hEvtInput);
        AssertRC (rc);
    }
    else
    {
        if (m_fConnectionsEnabled)
        {
            m_fConnectionsEnabled = false;

            /** @todo Disconnect existing clients and shutdown the transport. */
        }
    }

    return VINF_SUCCESS;
}

int VRDPServer::Disconnect (uint32_t u32ClientId, bool fReconnect)
{
    VRDPClient *pClient = m_clientArray.ThreadContextGetClient (u32ClientId, VRDP_CONTEXT_VM);

    if (!pClient)
    {
        return VERR_INVALID_PARAMETER;
    }

    pClient->NotifyDisconnect (fReconnect);

    pClient->ThreadContextRelease (VRDP_CONTEXT_VM);

    return VINF_SUCCESS;
}

int VRDPServer::Redirect (uint32_t u32ClientId,
                          const char *pszServer,
                          const char *pszUser,
                          const char *pszDomain,
                          const char *pszPassword,
                          uint32_t u32SessionId,
                          const char *pszCookie)
{
    VRDPClient *pClient = m_clientArray.ThreadContextGetClient (u32ClientId, VRDP_CONTEXT_VM);

    if (!pClient)
    {
        return VERR_INVALID_PARAMETER;
    }

    pClient->Redirect(pszServer,
                      pszUser,
                      pszDomain,
                      pszPassword,
                      u32SessionId,
                      pszCookie);

    pClient->ThreadContextRelease (VRDP_CONTEXT_VM);

    return VINF_SUCCESS;
}


int VRDPServer::StartThread (RTTHREAD *pthread, PFNRTTHREAD func, const char *pszThreadName)
{
    RTTHREAD thread = NIL_RTTHREAD;

    VRDPServerThreadStartCtx ctx;

    ctx.rc = VINF_SUCCESS;
    ctx.pVRDPServer = this;

    int rc = RTThreadCreate (&thread, func, &ctx, 0 /* default stack size */,
                             RTTHREADTYPE_VRDP_IO, RTTHREADFLAGS_WAITABLE, pszThreadName);

    if (RT_SUCCESS(rc))
    {
        /* Wait until the thread is ready. */
        rc = RTThreadUserWait (thread, 60000);
        AssertRC (rc);

        if (RT_SUCCESS (rc))
        {
            /* The wait was successful, get the rc from the thread. */
            rc = ctx.rc;
#ifdef DEBUG_sunlover
            AssertRC (rc);
#endif /* DEBUG_sunlover */
        }

        if (rc == VINF_VRDP_THREAD_STARTED)
        {
            /* The thread started successfully. */
            SERVERLOG(("Thread %s started\n", pszThreadName));
            *pthread = thread;
        }
        else
        {
            /* The thread does not want to start. Ignore rc here. */
            RTThreadWait (thread, 60000, NULL);
            *pthread = NIL_RTTHREAD;
        }
    }

    if (RT_FAILURE (rc))
    {
        VRDPLOGREL(("Failed to start %s thread, rc = %Rrc\n", pszThreadName, rc));
    }

    return rc;
}


void VRDPServer::notifyThreadStarted (RTTHREAD self, VRDPServerThreadStartCtx *pCtx, int rc)
{
    Assert (pCtx);
    Assert (pCtx->rc == VINF_SUCCESS);
    Assert (pCtx->pVRDPServer == this);

    /* Inform the server that the thread has started. */
    pCtx->rc = RT_SUCCESS(rc)?
                   VINF_VRDP_THREAD_STARTED:
                   rc;

    rc = RTThreadUserSignal (self);
    AssertRC (rc);
}


void VRDPServer::ShutdownThreads (void)
{
    int rc;

    SERVERLOG(("ShutdownThreads\n"));

    m_fShutdownThreads = true;

    SERVERLOG(("Disconnect all clients.\n"));

    uint32_t u32ClientId = 0;
    VRDPClient *pClient = NULL;

    while ((pClient = m_clientArray.ThreadContextGetNextClient(&u32ClientId, VRDP_CONTEXT_VM)) != NULL)
    {
        pClient->NotifyDisconnect (/* fReconnect = */  false);

        pClient->ThreadContextRelease (VRDP_CONTEXT_VM);
    }

    if (m_outputThread != NIL_RTTHREAD)
    {
        RaiseOutputEvent ();

        rc = RTThreadWait (m_outputThread, 60000, NULL);

        /* If the thread has terminated, the handle will be set to NIL. */
        if (m_outputThread != NIL_RTTHREAD)
        {
            VRDPLOGREL(("Failed to stop the VRDP output thread rc = %Rrc!!!\n", rc));
        }

        if (rc == VERR_INVALID_HANDLE)
        {
            rc = VINF_SUCCESS;
        }

        AssertRC (rc);

        SERVERLOG(("Output thread has been shut down\n"));
    }

    if (m_pTransport)
    {
        /* That is also a signal to input thread to terminate. */
        m_pTransport->NotifyShutdown();
    }

    if (m_inputThread != NIL_RTTHREAD)
    {
        /* Kick the input thread out of any out loop "sleeps".  */
        rc = RTSemEventSignal (m_hEvtInput);
        AssertRC (rc);

        /* Wait for the input thread termination. */
        rc = RTThreadWait (m_inputThread, 60000, NULL);

        /* If the thread has terminated, the handle will be set to NIL. */
        if (m_inputThread != NIL_RTTHREAD)
        {
            VRDPLOGREL(("Failed to stop the VRDP input thread rc = %Rrc\n", rc));
        }

        if (rc == VERR_INVALID_HANDLE)
        {
            rc = VINF_SUCCESS;
        }

        AssertRC (rc);

        SERVERLOG(("Input thread has been shut down\n"));
    }

    SERVERLOG(("ShutdownThreads completed.\n"));
}

/*
 * @thread VM.
 */
void VRDPServer::ProcessResize (void)
{
    /* This is called on VM thread when a framebuffer has been resized.
     * Resize the shadowbuffers, so the VM thread can process
     * bitmap updates and orders. And the shadowbuffers will generate
     * an RESIZE action for clients.
     * If a screen does not have a framebuffer, then the corresponding shadow buffer
     * is deleted.
     */
    unsigned uScreenId;

    for (uScreenId = 0; uScreenId < m_cMonitors; uScreenId++)
    {
        /* Get the referenced, locked framebuffer pointer and position for the current monitor. */
        VRDPBITSRECT bitsRect;
        memset (&bitsRect, 0, sizeof (VRDPBITSRECT));

        shadowBufferSetAccessible(uScreenId, false);

        bool fAvailable = FrameBufferQueryLock (&bitsRect, NULL, uScreenId);

        /* The current information. */
        VRDPFBINFO *pFBInfo = &m_paFBInfos[uScreenId];

        SERVERLOG(("uScreenId = %d, fAvailable = %d\n", uScreenId, fAvailable));

        if (!fAvailable)
        {
            shadowBufferResize (uScreenId, NULL, VRDP_TRANSFORM_ROTATE_0);
        }
        else
        {
            /* Check whether the current configuration is the same as the new one. */
            if (memcmp (&bitsRect, &pFBInfo->bitsRect, sizeof (VRDPBITSRECT)) != 0)
            {
                shadowBufferResize (uScreenId, &bitsRect, VRDP_TRANSFORM_ROTATE_0);
            }
        }

        /* Update the information. */
        pFBInfo->bitsRect = bitsRect;

        if (fAvailable)
        {
            appFramebufferUnlock (ApplicationCallbacks (), ApplicationCallbackPointer (), uScreenId);
        }

        shadowBufferSetAccessible(uScreenId, true);
    }
}

/** @thread VM */
void VRDPServer::ProcessBitmapUpdate (unsigned uScreenId, unsigned x, unsigned y, unsigned w, unsigned h)
{
    SERVERLOG(("BITMAP %d: %d,%d %dx%d\n", uScreenId, x, y, w, h));

    /* Copy FB content to shadow buffers. Shadow buffer + MSB keep track of
     * dirty area and output thread asks shadow buffer about the dirty region.
     */
    shadowBufferBitmapUpdate (uScreenId, x, y, w, h);
}

void VRDPServer::ProcessOutputUpdate (unsigned uScreenId, void *pvUpdate, unsigned cbUpdate)
{
    SERVERLOG(("ORDER %d: cb %d\n", uScreenId, cbUpdate));

    /* The shadow buffer keeps track of orders and provides them to the output thread.
     */
    shadowBufferOrder (uScreenId, pvUpdate, cbUpdate);
}

static void redrawClientRectangle(unsigned uScreenId,
                                  const RGNRECT *pRectClient,
                                  VRDPClient *pClient,
                                  bool fForceClearClientRect)
{
    RGNRECT rectShadowBuffer;

    /* Convert the client rect to shadow buffer coords. */
    pClient->TP ()->DesktopMap ()->Client2Screen (uScreenId, pRectClient, &rectShadowBuffer);

    /* Do a CLS if it was requested or the client area exceeds the shadow buffer area. */
    bool fClearClientRect =    fForceClearClientRect
                            || (   pRectClient->w != rectShadowBuffer.w
                                || pRectClient->h != rectShadowBuffer.h);

    shadowBufferRedrawUpdate (uScreenId,
                              &rectShadowBuffer,
                              fClearClientRect? pRectClient: NULL); /* Where to do a CLS. */
}

void VRDPServer::ProcessUpdateComplete (void)
{
    /* That is called when the sequence of updates has been completed and
     * therefore the framebuffer image is synchronized with previous
     * updates.
     *
     * Check whether clients have pending redraw requests or fullscreen updates.
     *
     * These updates are processed for every client separately.
     *
     */
    shadowBufferUpdateComplete();

    uint32_t u32ClientId = 0;
    VRDPClient *pClient = NULL;

    while ((pClient = m_clientArray.ThreadContextGetNextClient (&u32ClientId, VRDP_CONTEXT_VM)) != NULL)
    {
        RGNRECT rectClient;
        unsigned uScreenId;
        VRDPRedrawInfo redraw;

        int rc = pClient->QueryRedraw (&redraw);

        if (RT_SUCCESS(rc))
        {
            bool fMultiMonitor = pClient->IsMultiMonitor ();

            if (redraw.fFullscreen)
            {
                /* There is a fullscreen update. Do it and ignore other updates. */
                if (fMultiMonitor)
                {
                    for (uScreenId = 0; uScreenId < m_cMonitors; uScreenId++)
                    {
                        /* Get the rectangle of the current monitor in client's coords. */
                        pClient->TP ()->DesktopMap ()->QueryClientMonitorRect (uScreenId, &rectClient);

                        redrawClientRectangle(uScreenId, &rectClient, pClient, true);
                    }
                }
                else
                {
                    uScreenId = pClient->TP ()->QueryScreenId ();

                    /* Get the rectangle of the current monitor in client's coords. */
                    pClient->TP ()->DesktopMap ()->QueryClientMonitorRect (uScreenId, &rectClient);

                    redrawClientRectangle(uScreenId, &rectClient, pClient, true);
                }
            }
            else
            {
                /* First process client rects. */
                int i;

                for (i = 0; i < redraw.cRectsClient; i++)
                {
                    TESTLOG(("REDRAW: [%d] client %d,%d %dx%d\n",
                                  i,
                                  redraw.aRectsClient[i].rectRedraw.x,
                                  redraw.aRectsClient[i].rectRedraw.y,
                                  redraw.aRectsClient[i].rectRedraw.w,
                                  redraw.aRectsClient[i].rectRedraw.h
                           ));

                    /* These updates are requested by the client. For rdesktop clear the area before repainting it.
                     * This is a workaround for a rdesktop issue.
                     */
                    bool fForceClearClientRect = pClient->TP ()->IsClientResolution()? false: true;

                    if (fMultiMonitor)
                    {
                        for (uScreenId = 0; uScreenId < m_cMonitors; uScreenId++)
                        {
                            /* Obtain the part of the client rect for the current monitor. */
                            pClient->TP ()->DesktopMap ()->Client2Client (uScreenId, &redraw.aRectsClient[i].rectRedraw, &rectClient);

                            redrawClientRectangle(uScreenId, &rectClient, pClient, fForceClearClientRect);
                        }
                    }
                    else
                    {
                        uScreenId = pClient->TP ()->QueryScreenId ();

                        /* Obtain the part of the client rect for the current monitor. */
                        pClient->TP ()->DesktopMap ()->Client2Client (uScreenId, &redraw.aRectsClient[i].rectRedraw, &rectClient);

                        redrawClientRectangle(uScreenId, &rectClient, pClient, fForceClearClientRect);
                    }
                }

                /* Now process shadowbuffer rects. */
                for (i = 0; i < redraw.cRectsShadowBuffer; i++)
                {
                    TESTLOG(("REDRAW: [%d] shadowbuffer %d,%d %dx%d @%d\n",
                                  i,
                                  redraw.aRectsShadowBuffer[i].rectRedraw.x,
                                  redraw.aRectsShadowBuffer[i].rectRedraw.y,
                                  redraw.aRectsShadowBuffer[i].rectRedraw.w,
                                  redraw.aRectsShadowBuffer[i].rectRedraw.h,
                                  redraw.aRectsShadowBuffer[i].uScreenId
                           ));

                    uScreenId = redraw.aRectsShadowBuffer[i].uScreenId;

                    /* Do not clear the area before repainting to avoid flickering. */
                    shadowBufferRedrawUpdate (uScreenId,
                                              &redraw.aRectsShadowBuffer[i].rectRedraw,
                                              NULL); /* No CLS. */
                }
            }

            ASMAtomicCmpXchgU32(&m_u32OrderFallbackStatus, VRDP_OF_STATUS_REDRAW_COMPLETED, VRDP_OF_STATUS_REDRAW_PENDING);
        }

        pClient->ThreadContextRelease (VRDP_CONTEXT_VM);
    }
}

void VRDPServer::RaiseOutputEvent (void)
{
    if (m_outputsem)
    {
        RTSemEventMultiSignal (m_outputsem);
    }
}

int VRDPServer::PostOutput (int iCode, uint32_t u32TargetClientId, const void *pvData, unsigned cbData)
{
    if (m_fShutdownThreads)
    {
        return VERR_NOT_SUPPORTED;
    }

    OutputUpdate *pUpdate = m_outputQueue.CreateUpdate (iCode, u32TargetClientId, pvData, cbData);

    if (!pUpdate)
    {
        return VERR_NO_MEMORY;
    }

    m_outputQueue.InsertUpdate (pUpdate);

    return VINF_SUCCESS;
}

int VRDPServer::PostOutputEvent (int iCode, uint32_t u32TargetClientId, const void *pvData, unsigned cbData)
{
    int rc = PostOutput (iCode, u32TargetClientId, pvData, cbData);
    if (RT_SUCCESS(rc))
    {
        RaiseOutputEvent ();
    }
    return rc;
}

/* @return Referenced, locked pointer to the framebuffer. */
bool VRDPServer::FrameBufferQueryLock (VRDPBITSRECT *pBitsRect, RGNRECT *pRect, unsigned uScreenId)
{
    VRDEFRAMEBUFFERINFO info;
    memset (&info, 0, sizeof (info));

    bool fAvailable = appFramebufferQuery (ApplicationCallbacks (), ApplicationCallbackPointer (), uScreenId, &info);

    if (fAvailable)
    {
        if (info.cWidth == 0 || info.cHeight == 0)
        {
             /* Do not support zero size. */
             fAvailable = false;
        }
    }

    if (fAvailable)
    {
        appFramebufferLock (ApplicationCallbacks (), ApplicationCallbackPointer (), uScreenId);

        /* Now fill the information as requested by the caller. */
        pBitsRect->cbLine        = info.cbLine;
        pBitsRect->cBitsPerPixel = info.cBitsPerPixel;
        pBitsRect->cbPixel       = (info.cBitsPerPixel + 7) / 8;

        if (pRect == NULL)
        {
            /* Entire framebuffer was requested. Set the dimensions and origin of the framebuffer in the
             * guest virtual desktop coordinates.
             */
            pBitsRect->rect.x = info.xOrigin;
            pBitsRect->rect.y = info.yOrigin;
            pBitsRect->rect.w = info.cWidth;
            pBitsRect->rect.h = info.cHeight;

            pBitsRect->pu8Bits = info.pu8Bits;
        }
        else
        {
            /* A rectangle within framebuffer is requested.
             * Adjust the supplied rectangle to be in the buffer.
             */
            int xLeft   = pRect->x;
            int xRight  = pRect->x + pRect->w;
            int yTop    = pRect->y;
            int yBottom = pRect->y + pRect->h;

            if (xLeft > xRight) { int tmp = xLeft; xLeft = xRight; xRight = tmp; }
            if (yTop > yBottom) { int tmp = yTop; yTop = yBottom; yBottom = tmp; }

            if (xLeft < 0) xLeft = 0;
            if (xRight > (int)info.cWidth) xRight = info.cWidth;

            if (yTop < 0) yTop = 0;
            if (yBottom > (int)info.cHeight) yBottom = info.cHeight;

            pBitsRect->rect.x = xLeft;
            pBitsRect->rect.y = yTop;
            pBitsRect->rect.w = xRight - xLeft;
            pBitsRect->rect.h = yBottom - yTop;

            pBitsRect->pu8Bits = info.pu8Bits +
                                 pBitsRect->rect.y * pBitsRect->cbLine +
                                 pBitsRect->rect.x * pBitsRect->cbPixel;
        }
    }

    return fAvailable;
}

static void vrdpResetKeyboardModifiers (const VRDECALLBACKS_4 *pCallbacks, void *pvCallback, VRDPKeyboardModifiers *pModifiers)
{
    VRDEINPUTSCANCODE sc;

    SERVERLOG(("Reset Left S/C/A %d/%d/%d Right S/C/A %d/%d/%d\n",
               pModifiers->fLeftShift, pModifiers->fLeftCtrl, pModifiers->fLeftAlt,
               pModifiers->fRightShift, pModifiers->fRightCtrl, pModifiers->fRightAlt
             ));

    if (pModifiers->fLeftShift)
    {
        sc.uScancode = 0x2a | 0x80;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
    }
    if (pModifiers->fLeftCtrl)
    {
        sc.uScancode = 0x1d | 0x80;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
    }
    if (pModifiers->fLeftAlt)
    {
        sc.uScancode = 0x38 | 0x80;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
    }
    if (pModifiers->fRightShift)
    {
        sc.uScancode = 0x36 | 0x80;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
    }
    if (pModifiers->fRightCtrl)
    {
        sc.uScancode = 0xe0;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
        sc.uScancode = 0x1d | 0x80;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
    }
    if (pModifiers->fRightAlt)
    {
        sc.uScancode = 0xe0;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
        sc.uScancode = 0x38 | 0x80;
        appInput (pCallbacks, pvCallback, VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
    }

    memset (pModifiers, 0, sizeof (VRDPKeyboardModifiers));
}

static void vrdpProcessKeyboardModifiers (VRDPDesktop::InputMsg *pmsg, VRDPKeyboardModifiers *pModifiers)
{
    bool fDown = (pmsg->device_flags & KBD_FLAG_UP) == 0;

    switch (pmsg->param1)
    {
        case 0x1d: /* Ctrl */
        {
            if (pmsg->device_flags & KBD_FLAG_EXT)
            {
                /* Right Ctrl */
                pModifiers->fRightCtrl = fDown;
            }
            else
            {
                /* Left Ctrl */
                pModifiers->fLeftCtrl = fDown;
            }
        } break;
        case 0x2a: /* Left Shift */
        {
            pModifiers->fLeftShift = fDown;
        } break;
        case 0x36: /* Right Shift */
        {
            pModifiers->fRightShift = fDown;
        } break;
        case 0x38: /* Alt */
        {
            if (pmsg->device_flags & KBD_FLAG_EXT)
            {
                /* Right Alt */
                pModifiers->fRightAlt = fDown;
            }
            else
            {
                /* Left Alt */
                pModifiers->fLeftAlt = fDown;
            }
        } break;
    }
}

/* virtual */ void VRDPServer::ProcessInput (unsigned uScreenId, VRDPDesktop::InputMsg *pmsg)
{
    SERVERLOG(("VRDPServer::ProcessInput: type %x, uScreenId = %d\n", pmsg->message_type, uScreenId));

    switch (pmsg->message_type)
    {
        case RDP_INPUT_SYNCHRONIZE:
        {
            SERVERLOG(("VRDPServer::ProcessInput: RDP_INPUT_SYNCHRONIZE: p1 %04x p2 %04x devflag %04x\n",
                        pmsg->param1, pmsg->param2, pmsg->device_flags));

            VRDEINPUTSYNCH synch;
            synch.uLockStatus = 0;

            if (pmsg->param1 & KBD_FLAG_SCROLL)
            {
                synch.uLockStatus |= VRDE_INPUT_SYNCH_SCROLL;
            }

            if (pmsg->param1 & KBD_FLAG_NUMLOCK)
            {
                synch.uLockStatus |= VRDE_INPUT_SYNCH_NUMLOCK;
            }

            if (pmsg->param1 & KBD_FLAG_CAPITAL)
            {
                synch.uLockStatus |= VRDE_INPUT_SYNCH_CAPITAL;
            }

            appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_SYNCH, &synch, sizeof (VRDEINPUTSYNCH));

            vrdpResetKeyboardModifiers (ApplicationCallbacks (), ApplicationCallbackPointer (), &m_KeyboardModifiers);
        } break;

        case RDP_INPUT_CODEPOINT:
        {
            /// @todo rdesktop does not use this message
        } break;

        case RDP_INPUT_VIRTKEY:
        {
            SERVERLOG(("VRDPServer::ProcessInput: virtkey %x, flags %x, param2 %x. virtual keys are unsupported!!!\n", pmsg->param1, pmsg->device_flags, pmsg->param2));

            /* 20060213 All supported clients use scancodes, therefore, there is not need for VK messages. */
        } break;

        case RDP_INPUT_SCANCODE:
        {
            // RDP client sends a scancode
            int keycode = pmsg->param1;

            SERVERLOG(("VRDPServer::ProcessInput: keycode 0x%x, flags 0x%x\n", pmsg->param1, pmsg->device_flags));

            vrdpProcessKeyboardModifiers (pmsg, &m_KeyboardModifiers);

            if (m_fKeyboardExtension)
            {
                /* Special processing for Ctrl-Del and Ctrl-R combinations. */
                if (keycode == 0x53)
                {
                    /* Del */
                    if (m_KeyboardModifiers.fRightCtrl)
                    {
                        /* Translate RCtrl-Del to Ctrl-Alt-Del */
                        appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_CAD, NULL, 0);
                    }
                }
                else if (keycode == 0x13)
                {
                    /* R */
                    if (m_KeyboardModifiers.fRightCtrl)
                    {
                        /* Translate RCtrl-R to VM Reset. */
                        appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_RESET, NULL, 0);
                    }
                }
#ifdef MEMLEAK
                else if (keycode == 0x32)
                {
                    /* M */
                    if (m_KeyboardModifiers.fRightCtrl)
                    {
                        /* Translate RCtrl-M to log rel memory dump. */
                        VRDPMemDump();
                    }
                }
#endif /* MEMLEAK */
            }

            /* now send the key code */
            if (pmsg->device_flags & KBD_FLAG_EXT)
            {
                VRDEINPUTSCANCODE sc;
                sc.uScancode = 0xe0;
                appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
            }

            if (pmsg->device_flags & KBD_FLAG_EXT2)
            {
                VRDEINPUTSCANCODE sc;
                sc.uScancode = 0xe1;
                appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
            }

            if (pmsg->device_flags & KBD_FLAG_UP)
            {
                /* Set the high bit which indicates the key up. */
                VRDEINPUTSCANCODE sc;
                sc.uScancode = keycode | 0x80;
                appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
            }
            else
            {
                /* Do not clear the high bit of the scancode. Some clients send such scancodes directly. */
                VRDEINPUTSCANCODE sc;
                sc.uScancode = keycode;
                appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_SCANCODE, &sc, sizeof (VRDEINPUTSCANCODE));
            }

            notifyInputActivity ();
        } break;

        case RDP_INPUT_MOUSE:
        {
#if 0
           /* Debug code. */
           if (pmsg->device_flags & MOUSE_FLAG_DOWN)
           {
                VRDEINPUTPOINT ap[] =
                {
                    /* .x,  .y, .uButtons */
                    { 215 + 0, 129 + 0, 0 },
                    { 215 + 0, 130 + 0, 0 },
                    { 215 + 0, 130 + 0, VRDE_INPUT_POINT_BUTTON1 },
                    { 215 + 1, 130 - 1, VRDE_INPUT_POINT_BUTTON1 },
                    { 215 + 5, 130 - 7, VRDE_INPUT_POINT_BUTTON1 },
                    { 215 + 5, 130 - 7, 0 }
                };

                int i;
                for (i = 0; i < RT_ELEMENTS(ap); i++)
                {
                    appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_POINT, &ap[i], sizeof (VRDEINPUTPOINT));
                    VRDPThreadSleep(1000);
                }
                break;
            }
#endif
            // client sends absolute mouse coordinates
            int dx = 0;
            int dy = 0;

            SERVERLOG(("VRDPServer::ProcessInput: flags 0x%x, param1 0x%x, param2 0x%x\n", pmsg->device_flags, pmsg->param1, pmsg->param2));

            if (pmsg->device_flags & (MOUSE_FLAG_MOVE | MOUSE_FLAG_DOWN))
            {
                /* The mouse position should be updated to the location specified by the xPos and yPos fields.
                 * Or a click event has occurred at the position specified by the xPos and yPos fields.
                 */
                dx = pmsg->param1 - m_mousex;
                m_mousex = pmsg->param1;

                dy = pmsg->param2 - m_mousey;
                m_mousey = pmsg->param2;
            }

            SERVERLOG(("VRDPServer::ProcessInput: flags 0x%x\n", pmsg->device_flags));

            if (pmsg->device_flags & MOUSE_FLAG_DOWN)
            {
                if (pmsg->device_flags & MOUSE_FLAG_BUTTON1)
                {
                    m_mouseButtons |= VRDE_INPUT_POINT_BUTTON1;
                }

                if (pmsg->device_flags & MOUSE_FLAG_BUTTON2)
                {
                    m_mouseButtons |= VRDE_INPUT_POINT_BUTTON2;
                }

                if (pmsg->device_flags & MOUSE_FLAG_BUTTON3)
                {
                    m_mouseButtons |= VRDE_INPUT_POINT_BUTTON3;
                }
            }
            else
            {
                if (pmsg->device_flags & MOUSE_FLAG_BUTTON1)
                {
                    m_mouseButtons &= ~VRDE_INPUT_POINT_BUTTON1;
                }

                if (pmsg->device_flags & MOUSE_FLAG_BUTTON2)
                {
                    m_mouseButtons &= ~VRDE_INPUT_POINT_BUTTON2;
                }

                if (pmsg->device_flags & MOUSE_FLAG_BUTTON3)
                {
                    m_mouseButtons &= ~VRDE_INPUT_POINT_BUTTON3;
                }
            }

            /* Wheel processing. rdesktop uses 0x0280 for "up" and 0x0380 for "down"
             * MSFT client uses 0x0278 for "up" and 0x0388 for "down". The server
             * detects the event by the 0x0300 mask.
             */
            if ((pmsg->device_flags & MOUSE_FLAG_WHEEL_MASK) == (MOUSE_FLAG_BUTTON4 & MOUSE_FLAG_WHEEL_MASK))
            {
                m_mouseButtons |= VRDE_INPUT_POINT_WHEEL_UP;
            }
            else
            {
                m_mouseButtons &= ~VRDE_INPUT_POINT_WHEEL_UP;
            }

            if ((pmsg->device_flags & MOUSE_FLAG_WHEEL_MASK) == (MOUSE_FLAG_BUTTON5 & MOUSE_FLAG_WHEEL_MASK))
            {
                m_mouseButtons |= VRDE_INPUT_POINT_WHEEL_DOWN;
            }
            else
            {
                m_mouseButtons &= ~VRDE_INPUT_POINT_WHEEL_DOWN;
            }

            SERVERLOG(("MOUSE: absolute (%d,%d), relative (%d,%d) buttons %8.8X\r\n", m_mousex, m_mousey, dx, dy, m_mouseButtons));

            VRDEINPUTPOINT p;
            p.x = m_mousex;
            p.y = m_mousey;
            p.uButtons = m_mouseButtons;

            /* Adjust mouse position for corresponding screen id. */
            shadowBufferMapMouse (uScreenId, &p.x, &p.y);

            appInput (ApplicationCallbacks (), ApplicationCallbackPointer (), VRDE_INPUT_POINT, &p, sizeof (VRDEINPUTPOINT));

            notifyInputActivity ();
        } break;
    }
    return;
}

/*
 * The VRDPServer external callbacks.
 */
int VRDPServer::ClientLogon (VRDPClient *pClient, const char *pszUser, const char *pszPassword, const char *pszDomain)
{
    Assert (IsInputThread ());

    SERVERLOG(("VRDPServer::ClientLogon [%s], [%s], [%s]\n", pszUser, pszPassword, pszDomain));

    int rc = appClientLogon (ApplicationCallbacks (), ApplicationCallbackPointer (), pClient->Id(), pszUser, pszPassword, pszDomain);

    SERVERLOG(("VRDPServer::ClientLogon callback returned %Rrc\n", rc));

    return rc;
}

/*
 * The VRDPServer external callbacks.
 */
int VRDPServer::ClientLocationInfo (VRDPClient *pClient,
                                    const char *pszCName,
                                    const char *pszCIPAddr,
                                    const char *pszCLocation,
                                    const char *pszCOtherInfo)
{
    Assert (IsInputThread ());
    int rc = VINF_SUCCESS;

    const char* infoMap[][2] = {
                                {pszCName, "NAME="},
                                {pszCIPAddr, "CIPA="},
                                {pszCLocation, "CLOCATION="},
                                {pszCOtherInfo, "COINFO="}
                               };
    unsigned int idx;

    for (idx = 0; idx < RT_ELEMENTS(infoMap); idx++)
    {
        if (infoMap[idx][0] == NULL)
            break;

        const char *pszInfoName = infoMap[idx][0];
        const char *pszPrefix = infoMap[idx][1];

        const uint32_t cchInfoName = (uint32_t)strlen(pszInfoName);
        const uint32_t cchPrefix = (uint32_t)strlen(pszPrefix);

        const uint32_t cchStatus = cchPrefix + cchInfoName;

        uint32_t cbAlloc = sizeof(VRDECLIENTSTATUS) + cchStatus;

        VRDECLIENTSTATUS *p = (VRDECLIENTSTATUS *)VRDPMemAlloc(cbAlloc);

        if (!p)
        {
            rc = VERR_NO_MEMORY;
            break;
        }

        p->u32ClientId = pClient->Id();
        p->cbStatus = cchStatus + 1;
        memcpy(p->achStatus, pszPrefix, cchPrefix);
        memcpy(&p->achStatus[cchPrefix], pszInfoName, cchInfoName + 1);

        rc = appProperty(ApplicationCallbacks(), ApplicationCallbackPointer(),
                         VRDE_SP_CLIENT_STATUS,
                         p, cbAlloc, NULL);

        VRDPMemFree(p);

        if (RT_FAILURE (rc))
        {
            SERVERLOG(("VRDPServer::Failed property Info [%s]\n", pszInfoName));
            break;
        }
    }

    SERVERLOG(("VRDPServer::ClientLocationInfo callback returned %Rrc\n", rc));
    return rc;
}

int VRDPServer::ClientAttach(VRDPClient *pClient, bool fAttached)
{
    Assert(IsInputThread ());

    SERVERLOG(("VRDPServer::ClientAttach [%d]\n", fAttached));

    const char *pszStatus = fAttached? "ATTACH": "DETACH";
    const uint32_t cchStatus = (uint32_t)strlen(pszStatus);

    uint32_t cbAlloc = sizeof(VRDECLIENTSTATUS) + cchStatus;

    VRDECLIENTSTATUS *p = (VRDECLIENTSTATUS *)VRDPMemAlloc(cbAlloc);

    if (!p)
    {
        return VERR_NO_MEMORY;
    }

    p->u32ClientId = pClient->Id();
    p->cbStatus = cchStatus + 1;
    memcpy(p->achStatus, pszStatus, cchStatus + 1);

    int rc = appProperty(ApplicationCallbacks(), ApplicationCallbackPointer(),
                         VRDE_SP_CLIENT_STATUS,
                         p, cbAlloc, NULL);

    SERVERLOG(("VRDPServer::ClientAttach callback returned %Rrc\n", rc));

    VRDPMemFree(p);

    return rc;
}

void VRDPServer::ClientConnect (VRDPClient *pClient)
{
    Assert (IsInputThread ());

    appClientConnect (ApplicationCallbacks (), ApplicationCallbackPointer (), pClient->Id());
}

void VRDPServer::ClientDisconnect (VRDPClient *pClient)
{
    Assert (IsInputThread ());

    VRDPMemDump();

    appClientDisconnect (ApplicationCallbacks (), ApplicationCallbackPointer (), pClient->Id(), pClient->QueryIntercepted ());
}

int VRDPServer::InterceptChannel (VRDPClient *pClient, uint32_t fu32Intercept, void **ppvIntercept)
{
    Assert (IsInputThread ());

    int rc = appIntercept (ApplicationCallbacks (), ApplicationCallbackPointer (), pClient->Id(), fu32Intercept, ppvIntercept);

    if (RT_SUCCESS (rc))
    {
        pClient->NotifyIntercepted (fu32Intercept);
    }

    return rc;
}


void VRDPServer::NotifyClientResolution (int w, int h, unsigned uScreenId)
{
    SERVERLOG((": VRDPServer::NotifyClientResolution %d, %d, at %d\n", w, h, uScreenId));
    /* Select the lowest resolution of all connected clients with the same screen id. */
    VRDP_ASSERT_CONTEXT_INPUT(this);

    VRDPClient *pClient = NULL;

    while ((pClient = m_clientArray.GetNextClient (pClient)) != NULL)
    {
        if (!pClient->IsClientResizable ()
            && pClient->TP()->DesktopMap() != NULL
            && pClient->IsScreenIdMatched (uScreenId))
        {
            /* Only for clients which do not support resizing. */
            RGNRECT rect;

            pClient->TP()->DesktopMap()->QueryClientMonitorRect (uScreenId, &rect);

            SERVERLOG((": VRDPServer::NotifyClientResolution client %d, %d\n", rect.w, rect.h));
            if (w > (int)rect.w)
            {
                w = rect.w;
            }

            if (h > (int)rect.h)
            {
                h = rect.h;
            }
        }
    }

    SERVERLOG(("VRDPServer::NotifyClientResolution adjusted to %d, %d\n", w, h));

    shadowBufferTransformWidthHeight (uScreenId, (unsigned *)&w, (unsigned *)&h);

/// @todo find out why the videomode hints are ignored.
//       The VMMDev host code filters them out. How it worked before?
//       May be the guest additions earlier detected that the hint is not the same as the resolution.
//       and now always wait for the VMMDEV_EVENT?
//
//       The same VM in VBox trunk without recent VRDP changes also behaves the same way.
//
    appVideoModeHint(ApplicationCallbacks (), ApplicationCallbackPointer (), w, h, 0, uScreenId);
}

void VRDPServer::QueryInfo (uint32_t index, void *pvBuffer, uint32_t cbBuffer, uint32_t *pcbOut)
{
    /* Assign the output size to 'no value was returned'. */
    *pcbOut = 0;

    /* How many bytes is required in the output buffer. */
    uint32_t cbOut;

    switch (index)
    {
        /* Number of active clients. */
        case VRDE_QI_ACTIVE:
        {
            cbOut = sizeof (uint32_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint32_t *)pvBuffer = m_clientArray.NumberOfClientsActive ();
            }
        } break;

        /* Number of clients which had a connection to the server. */
        case VRDE_QI_NUMBER_OF_CLIENTS:
        {
            cbOut = sizeof (uint32_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint32_t *)pvBuffer = m_clientArray.NumberOfClientsInactive ();
            }
        } break;

        /* When last connection was established. */
        case VRDE_QI_BEGIN_TIME:
        {
            cbOut = sizeof (int64_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(int64_t *)pvBuffer = m_clientArray.TimeLastConnect ();
            }
        } break;

        /* When last connection was terminated or current time if connection still active. */
        case VRDE_QI_END_TIME:
        {
            cbOut = sizeof (int64_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(int64_t *)pvBuffer = m_clientArray.TimeLastDisconnect ();
            }
        } break;

        /* How many bytes were sent in last connected connection. */
        case VRDE_QI_BYTES_SENT:
        {
            cbOut = sizeof (uint64_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint64_t *)pvBuffer = m_clientArray.BytesSentLast ();
            }
        } break;

        case VRDE_QI_BYTES_SENT_TOTAL:
        {
            cbOut = sizeof (uint64_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint64_t *)pvBuffer = m_clientArray.BytesSentAll ();
            }
        } break;

        case VRDE_QI_BYTES_RECEIVED:
        {
            cbOut = sizeof (uint64_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint64_t *)pvBuffer = m_clientArray.BytesRecvLast ();
            }
        } break;

        case VRDE_QI_BYTES_RECEIVED_TOTAL:
        {
            cbOut = sizeof (uint64_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint64_t *)pvBuffer = m_clientArray.BytesRecvAll ();
            }
        } break;

        case VRDE_QI_USER:
        {
            *pcbOut = m_clientArray.StringLastUser (pvBuffer, cbBuffer);
        } break;

        case VRDE_QI_DOMAIN:
        {
            *pcbOut = m_clientArray.StringLastDomain (pvBuffer, cbBuffer);
        } break;

        case VRDE_QI_CLIENT_NAME:
        {
            *pcbOut = m_clientArray.StringLastClientName (pvBuffer, cbBuffer);
        } break;

        case VRDE_QI_CLIENT_IP:
        {
            *pcbOut = m_clientArray.StringLastClientIP (pvBuffer, cbBuffer);
        } break;

        case VRDE_QI_CLIENT_VERSION:
        {
            cbOut = sizeof (uint32_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint32_t *)pvBuffer = m_clientArray.NumberLastClientVersion ();
            }
        } break;

        case VRDE_QI_ENCRYPTION_STYLE:
        {
            cbOut = sizeof (uint32_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(uint32_t *)pvBuffer = m_clientArray.LastClientEncryptionStyle();
            }
        } break;

        case VRDE_QI_PORT:
        {
            cbOut = sizeof (int32_t);

            if (cbBuffer >= cbOut)
            {
                *pcbOut = cbOut;
                *(int32_t *)pvBuffer = m_u32BindPort;
            }
        } break;
    }

    return;
}

/* Allowed security methods. */
#define VRDP_SECURITY_ALLOW_RDP 0x00000001
#define VRDP_SECURITY_ALLOW_TLS 0x00000002

static uint32_t querySecurityMethodMask(VRDPServer *pServer)
{
    /* Default is to require TLS. */
    uint32_t u32Mask = VRDP_SECURITY_ALLOW_TLS;

    char *pszValue = NULL;
    uint32_t cbOut = 0;
    int rc = appFeature (pServer->ApplicationCallbacks (), pServer->ApplicationCallbackPointer (),
                         "Property/Security/Method",
                         &pszValue, &cbOut);

    if (RT_SUCCESS(rc) && pszValue)
    {
        if (RTStrICmp(pszValue, "RDP") == 0)
        {
            u32Mask = VRDP_SECURITY_ALLOW_RDP;
        }
        else if (RTStrICmp(pszValue, "TLS") == 0)
        {
            u32Mask = VRDP_SECURITY_ALLOW_TLS;
        }
        else if (RTStrICmp(pszValue, "NEGOTIATE") == 0)
        {
            u32Mask = VRDP_SECURITY_ALLOW_RDP
                    | VRDP_SECURITY_ALLOW_TLS;
        }
        else if (*pszValue != 0)
        {
            /* Not empty string value. Mention it. */
            VRDPLOGREL(("Unsupported 'Security/Method' = '%s'.\n",
                         pszValue));
        }

        VRDPMemFree(pszValue);
    }

    switch (u32Mask)
    {
        case VRDP_SECURITY_ALLOW_RDP | VRDP_SECURITY_ALLOW_TLS:
            VRDPLOGREL(("Negotiating security method with the client.\n"));
            break;
        case VRDP_SECURITY_ALLOW_RDP:
            VRDPLOGREL(("Standard RDP Security.\n"));
            break;
        case VRDP_SECURITY_ALLOW_TLS:
            VRDPLOGREL(("Enhanced RDP Security.\n"));
            break;
        default:
            VRDPLOGREL(("RDP security mask 0x%x\n.\n", u32Mask));
            break;
    }

    return u32Mask;
}

bool VRDPServer::SelectSecurityProtocol(uint32_t u32RequestedProtocols,
                                        uint32_t *pu32ResponseCode)
{
    /* Use a config parameter: that is which of 2 protocols is supported. */
    uint32_t u32Mask = querySecurityMethodMask(this);

    /* TLS is preferable. Check for it first. */
    if (u32RequestedProtocols & PROTOCOL_SSL)
    {
        /* The client supports TLS. */
        VRDPLOGREL(("The client requests Enhanced RDP Security.\n"));
        if (u32Mask & VRDP_SECURITY_ALLOW_TLS)
        {
            *pu32ResponseCode = PROTOCOL_SSL;
            return true;
        }

        *pu32ResponseCode = SSL_NOT_ALLOWED_BY_SERVER;
        return false;
    }

    if (u32RequestedProtocols == PROTOCOL_RDP)
    {
        VRDPLOGREL(("The client requests Standard RDP Security.\n"));
        /* The client supports RDP. */
        if (u32Mask & VRDP_SECURITY_ALLOW_RDP)
        {
            *pu32ResponseCode = PROTOCOL_RDP;
            return true;
        }

        *pu32ResponseCode = SSL_REQUIRED_BY_SERVER;
        return false;
    }

    /* The client sends flags which is unknown to the server. Tell it that TLS is required. */
    VRDPLOGREL(("The client requests unsupported RDP Security method 0x%x.\n", u32RequestedProtocols));
    *pu32ResponseCode = SSL_REQUIRED_BY_SERVER;
    return false;
}

/* virtual */ int VRDPServer::QueryFeature (const char *pszName, char **ppszValue, uint32_t *pcbOut)
{
    int rc = appFeature (ApplicationCallbacks (), ApplicationCallbackPointer (),
                         pszName,
                         ppszValue, pcbOut);
    return rc;
}
